package gov.va.genisis2.dao.impl;

import java.util.Date;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.dao.IUserRefreshHistoryDao;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.UserRefreshHistory;

/**
 * The Class UserRefreshHistoryDAO.
 *
 * The UserRefreshHistory data access object (DAO) is an object that provides an
 * abstract interface to some type of database or other persistence mechanism.
 * By mapping application calls to the persistence layer, UserRefreshHistory DAO
 * provide some specific data operations without exposing details of the
 * database.
 */
@Repository
@Transactional(value = "transactionManager")
public class UserRefreshHistoryDao extends AbstactHibernateDao implements IUserRefreshHistoryDao {

	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(UserRefreshHistoryDao.class);

	@Autowired
	private SessionFactory sessionFactory;

	@Override
	public Date getLatestUserRefreshHistory() throws GenisisDAOException {
		Date lastUserRefreshed = null;

		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Date> criteriaQuery = criteriaBuilder.createQuery(Date.class);
			Root<UserRefreshHistory> root = criteriaQuery.from(UserRefreshHistory.class);
			Predicate statusCheck = criteriaBuilder.equal(root.get(CommonEnum.STATUS.getText()), CommonEnum.STATUS_COMPLETED.getText());
			Predicate endDtNull = criteriaBuilder.isNotNull(root.get(CommonEnum.END_DATE.getText()));
			Predicate conditions = criteriaBuilder.and(statusCheck, endDtNull);
			criteriaQuery.where(conditions);
			
			criteriaQuery.select(criteriaBuilder.greatest(root.<Date>get(CommonEnum.END_DATE.getText())));
			lastUserRefreshed = session.createQuery(criteriaQuery).getSingleResult();
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETLATESTUSERREFRESHHISTORY.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}

		return lastUserRefreshed;
	}

	@Override
	public int createUserRefreshHistory(UserRefreshHistory userRefreshHistory) throws GenisisDAOException {
		
		if (userRefreshHistory == null) {
			throw new GenisisDAOException("UserRefreshHistory entity is null");
		}
		
		int id = 0; 
		
		try {
			Session session = sessionFactory.getCurrentSession();
			//TODO: remove - switch to use SEQUENCE
			id = getMaxRowValue(session, UserRefreshHistory.class, CommonEnum.ID_COLUMN.getText());
			userRefreshHistory.setId(id);
			id = createData(session, userRefreshHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception occured on createUserRefreshHistory.", ex);
			throw new GenisisDAOException("Exception occured on createUserRefreshHistory.", ex);
		}
		
		return id;
	}

	@Override
	public UserRefreshHistory updateUserRefreshHistory(UserRefreshHistory userRefreshHistory) throws GenisisDAOException {
		UserRefreshHistory updatedUserRefreshHistory;
		try {
			Session session = sessionFactory.getCurrentSession();
			updatedUserRefreshHistory = (UserRefreshHistory) updateData(session, userRefreshHistory); 
		} catch (Exception ex) {
			LOGGER.error("Exception occured on updateUserRefreshHistory.", ex);
			throw new GenisisDAOException("Exception occured on updateUserRefreshHistory.", ex);
		}
		
		return updatedUserRefreshHistory;
	}
	
	/**
	 * @param sessionFactory the sessionFactory to set
	 */
	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}
}